/*******************************************************************************
Author: John Iselin
Date Updated: September 26, 2019

Goldman School of Public Policy 
Masters in Public Policy Program
Graduate Student Researcher for Hilary Hoynes

Results do-file. 

This do-file produces the tabulations and regressions to explore the mobility of 
children born post-1968 in the United States, conditional on the availibility of 
food stamps in their county of birth. 

This results process has the following steps: 
(1) Take the data produced in the cleaning step and produce results variables 
(2) Produce summary statistics and tabulations 
(3) Run regression models

Any questions can be directed to John Iselin at johniselin@berkeley.edu

*******************************************************************************/


********************************************************************************
* (1) Take the Data Produced in the Cleaning Step and Produce Results Variables 
********************************************************************************

set matsize 1000

use ${data}psid_individual_wsens_fsp, clear

sort id year 

des

label define TAG 1 "Yes" 0 "No", replace 

**** Generate FSP variables 

** Any FSP in YOB
gen tag_fs_yob = (fs_yearYOB <= alt_yob) 

** bindex = index of birth year/month where 1968:1=1 [our first bith cohort]
gen bindex = (alt_yob - 1968) * 12 + mob

** pindex = index of year/month where FSP implemented, where 1968:1=1
gen pindex = (fs_yearYOB - 1968) * 12 + fs_monthYOB

** Share of early childhood (0-5) exposed to FSP, by county of birth
gen share_fsp_0_5 = min(max(bindex+5*12-pindex,0),5*12)/(5*12)
label variable share_fsp_0_5 	///
	"Share of early childhood (0-5) exposed to FSP, by county of birth."
replace share_fsp_0_5 = . if fs_yearYOB == .
tab alt_yob, summarize(share_fsp_0_5) means 

/*
* AGES 6 AND 18
gen share_fsp_cob_5_18 = min(max(bindex+18*12-pindex,0),13*12)/(13*12)
label variable share_fsp_cob_5_18 	///
	"Share of later childhood (5-18) exposed to FSP, by county of birth."
replace share_fsp_cob_5_18 = . if fs_yearYOB == .
tab alt_yob, summarize(share_fsp_cob_5_18) means
*/

** Any FSP in current year 
gen tag_fs_year = (fs_year <= year )

** Count of months of FSP exposure by year where age is between 0 and 5

gen months_fsp_year = .

* All other ages
replace months_fsp_year = 0 if alt_age >= 6

* FSP active for full year, Not YOB, Not Age 5
replace months_fsp_year = 12 if 					///
		fs_year < year & 							///
		alt_yob != year	&							///
		alt_age < 5
* FSP active for full year, YOB, Not Age 5	
replace months_fsp_year = 12 - (mob - 1) if 		///
		fs_year < year & 							///
		alt_yob == year
		
* FSP active for full year, Not YOB, Age 5		
replace months_fsp_year = mob -1 if 				///
		fs_year < year & 							///
		alt_yob != year	&							///
		alt_age == 5		

* FSP not active in year
replace months_fsp_year = 0 if 						///
		fs_year > year & 							///
		alt_age <= 5

* FSP active for partial year, Not YOB, Not Age 5		
replace months_fsp_year = 12 - (fs_month - 1) if 	///
		fs_year == year & 							///
		alt_yob != year & 							///
		alt_age < 5 

* FSP active for partial year, YOB, FSP Month <= MOB, Not Age 5				
replace months_fsp_year = 12 - (mob - 1) if 		///
		fs_year == year & 							///
		alt_yob == year & 							///
		fs_month <= mob 
		
* FSP active for partial year, YOB, FSP Month > MOB, Not Age 5				
replace months_fsp_year = 							///
		(12 - (mob - 1)) - (fs_month - mob) if 		///
		fs_year == year & 							///
		alt_yob == year & 							///
		fs_month > mob

* FSP active for partial year, Not YOB, Age 5, FSP Month <= MOB Month 				
replace months_fsp_year = mob - (fs_month - 1) if 	///
		fs_year == year & 							///
		alt_yob != year & 							///
		alt_age == 5 & 								///
		fs_month <= mob		
		
* FSP active for partial year, Not YOB, Age 5, FSP Month > MOB		
replace months_fsp_year = 0 if 						///
		fs_year == year & 							///
		alt_yob != year & 							///
		alt_age == 5 & 								///
		fs_month > mob				
		
tab months_fsp_year , m

gen temp = months_fsp_year if alt_age <= 5
replace temp = 0 if temp == .
by id: egen tot_months_fsp = total(temp)
tab tot_months_fsp, m
drop temp

** Years of early childhood observed  
gen temp = 12 if alt_age >= 1 & alt_age <= 4
replace temp = 12 - (mob - 1) if alt_age == 0
replace temp = (mob - 1) if alt_age == 5
replace temp = 0 if alt_age > 5

by id: egen child_months = total(temp)
tab child_months
drop temp

** Number of months alive pre-1968
gen alt_bindex = -bindex if bindex <= 0
replace alt_bindex = 0 if alt_bindex == .

** Number of months of exposure to FSP in COB pre-1968 
gen alt_pindex = -pindex if pindex <= 0
replace alt_pindex = 0 if alt_pindex == .
 
** Number of months of exposure of FSP in COB pre-1968 between 0-5
gen pre_68_fsp_ec_expose = min(min(alt_bindex,59), alt_pindex)
 
** Share of early childhood (0-5) exposed to FSP, by current counties
** Note: We have to make a number of assumptions to create this measure
**		(1) If you are born post-1974, we assume you have FSP for all 5 years
**		(2) If you are born prior to 1964, we use share_fsp_0_5
**		(3) If you are born in or after 1968, we calculate your exposure based 
**			on the share of your observed years between 0 and 5 years old. 
** 		(4) If you are born between 1964 and 1967, we use share_fsp_0_5 for 
**			unobserved years, and observed years for the rest. 
gen share_cur_fsp_0_5 = . 
label variable share_cur_fsp_0_5 				///
	"Share of early childhood (0-5) exposed to FSP, by actual counties."

* (1)	Equal to 1 if Born after 1974
replace share_cur_fsp_0_5 = 1 if 				///
		alt_yob >= 1975	
	
* (2) 	Equal to share_fsp_0_5 if born before 1964
replace share_cur_fsp_0_5 = share_fsp_0_5 if 	///
		alt_yob <= 1963 & 						///
		share_cur_fsp_0_5 == .
		
* (3) 	Equal to share of obs. months with FSP out of months observed before 5		
replace share_cur_fsp_0_5 = 					///
		tot_months_fsp / child_months if 		///
		alt_yob >= 1968 & 						///
		share_cur_fsp_0_5 == .
		
* (4) 	Equal to observed months with FSP + Pre-1968 months in COB divided by 
*		observed months before 5 + months alive before 1968
replace share_cur_fsp_0_5 = 						///
		(tot_months_fsp + pre_68_fsp_ec_expose)/	///
		(child_months + alt_bindex) if 				///
		alt_yob < 1968 & 							///
		alt_yob > 1963 & 							///	
		share_cur_fsp_0_5 == .

tab alt_yob, summarize(share_cur_fsp_0_5) means 

** Calculate the share of the year an observation is exposed to the FSP

gen fsp_expose = 0					//	No exposure 
label variable fsp_expose "Share of the year exposed to FPS"
replace fsp_expose = 1 				/// Full year exposure 
		if (fs_year < year )
replace fsp_expose = fs_month / 12 	/// Partial year exposure 
		if (fs_year == year )

** Generate Parental Education variable 
by id: egen mode_parental_education = mode(parental_education), max
gen tag_parent_hs = (mode_parental_education > 12)
drop mode_parental_education

** Generate County Mobility variable 
gen tag_mover = 			(countyfips != countyfipsYOB) |		///
							(stfips != stfipsYOB)
gen tag_mover_to_fsp = 		((countyfips != countyfipsYOB) |	///
							(stfips != stfipsYOB)) &			///
							(fs_year < year )
gen tag_mover_to_nofsp = 	((countyfips != countyfipsYOB) |	///
							(stfips != stfipsYOB)) &			///
							(fs_year > year )
gen tag_mover_from_fsp = 	((countyfips != countyfipsYOB) |	///
							(stfips != stfipsYOB)) &			///
							(fs_yearYOB <= alt_yob)
gen tag_mover_lose_fsp = 	((countyfips != countyfipsYOB) |	///
							(stfips != stfipsYOB)) &			///
							(fs_yearYOB <= alt_yob)	& 			///
							(fs_year > year)
gen tag_mover_gain_fsp = 	((countyfips != countyfipsYOB) |	///
							(stfips != stfipsYOB)) &			///
							(fs_yearYOB > alt_yob)	& 			///
							(fs_year <= year)
							
** Generate unique id for county+state of birth
egen cob_id = group(countyfipsYOB statenameYOB)
	
** Generate test for positive weight
gen tag_weight = (weight > 0)


** Label Variables 

label variable alt_yob "Cleaned year of birth"
label variable alt_age "Age calculated via cleaned year of birth"
label variable alt_age_bucket "Calculated age buckets"

label variable tag_weight 		///
	"Does the observation have positive weight?"
label values tag_weight TAG

label variable share_fsp_0_5	///
	"Share of Age 0-5 with FSP Access in Birth County"
	
label variable share_cur_fsp_0_5	///
	"Share of Age 0-5 with FSP Access in Current Counties"

label variable tag_fs_year "Does your current county have the FSP?"
label values tag_fs_year TAG

label variable tag_fs_yob "Did your first observed county have the FSP?"
label values tag_fs_yob TAG

label variable tag_parent_hs 	///
	"Did your mother (father) have more than a high school education?"
label values tag_parent_hs TAG

label variable tag_mover 		///
	"Moved Counties?"
label values tag_mover TAG

label variable tag_mover_to_fsp ///
	"Moved to County w. FSP access?" 
label values tag_mover_to_fsp TAG

label variable tag_mover_to_nofsp ///
	"Moved to County w/o FSP access?"

label variable tag_mover_from_fsp ///
	"Have you moved counties (from a county with FSP) since first observed?"	
label values tag_mover_from_fsp TAG

label variable tag_mover_lose_fsp ///
	"Have you moved counties (and lost access to the FSP) since first observed?"	
label values tag_mover_lose_fsp TAG		

label variable tag_mover_gain_fsp ///
	"Have you moved counties (and gained access to the FSP) since first observed?"	
label values tag_mover_gain_fsp TAG		
	
* Basic Tabulations
count
sum 


********************************************************************************
* (2) Run regression models  *************************************************** 
********************************************************************************		
				
drop if weight == 0 

*** Samples by YOB 
gen sample_all = (alt_yob <= 1970)
gen sample_post_65 = (alt_yob >= 1965) & (alt_yob <= 1970)
gen sample_post_68 = (alt_yob >= 1968) & (alt_yob <= 1970)

set emptycells drop
eststo clear 
local count = 0
local titles ""
** Loop over two dependent variables 
foreach dep of varlist tag_mover tag_mover_to_fsp {
	
	if "`dep'" == "tag_mover" {
		local dep_label "Moved"
	}
	else {
		local dep_label "Moved-to-FSP"
	}
	*** Loop over age groups 
	foreach age of numlist 5(5)15 {
		
		*** Loop over YOB Sample 
		foreach sample of varlist sample_all sample_post_65 sample_post_68 {
		
		if "`sample'" == "sample_all" {
			local yob_label "Full-Sample"
		}
		else if "`sample'" == "sample_post_65" {
			local yob_label "Post-1965"
		}
		else {
			local yob_label "Post-1968"
		}		
		
		local count = `count' + 1
		dis "Regression `count'"
		dis "`dep_label'"
		dis "Age == `age'"
		dis "`yob_label'"
		
			
		eststo:	reg `dep' share_fsp_0_5 i.cob_id i.alt_yob	///
				[aw = weight] if 							///
				alt_age == `age' & 							/// Age Restr.
				`sample' == 1								//	Sample Restr.
				
		local title`count' "(`dep_label')(`age')(`yob_label')"
		local titles `titles' `title`count''
		
		}		
	}
}

esttab using "${results}regression",  replace csv label					///
	b(%8.3fc) p(%8.3fc) keep(share_fsp_0_5) stats(r2 N , fmt(3 0))  ///
	mtitles (`titles')												///
	title(The effect of childhood FSP exposure on migration) 	
********************************************************************************
* (3) Tabulations and Statistics  **********************************************
********************************************************************************		
		
*** Scatterplot of Share of Months of FSP based on county of birth vs. based 
***	on current counties, by YOB 

scatter share_fsp_0_5 share_cur_fsp_0_5 						///
		if first_weight_obs == 1 & 								///
		sample_all == 1,										///
		title(Corr. Between Measures of Childhood FSP Exposure)	///
		subtitle(Full Sample)									///
		ytitle("Share of Childhood (0-5) Exposed to FSP, Birth County")  	///    								///
		xtitle("Share of Childhood (0-5) Exposed to FSP, Current Counties")         

graph export ${results}scatter_full_sample.pdf, as(pdf) replace
		
		
scatter share_fsp_0_5 share_cur_fsp_0_5 						///
		if first_weight_obs == 1 & 								///
		sample_post_65 == 1,									///
		title(Corr. Between Measures of Childhood FSP Exposure)	///
		subtitle(Sample Born During or After 1965)				///
		ytitle("Share of Childhood (0-5) Exposed to FSP, Birth County")  	///    								///
		xtitle("Share of Childhood (0-5) Exposed to FSP, Current Counties")   
graph export ${results}scatter_after_1965.pdf, as(pdf) replace		

scatter share_fsp_0_5 share_cur_fsp_0_5 						///
		if first_weight_obs == 1 & 								///
		sample_post_68 == 1,									///
		title(Corr. Between Measures of Childhood FSP Exposure)	///
		subtitle(Sample Born During or After 1968)				///
		ytitle("Share of Childhood (0-5) Exposed to FSP, Birth County")  	///    								///
		xtitle("Share of Childhood (0-5) Exposed to FSP, Current Counties")   

graph export ${results}scatter_after_1968.pdf, as(pdf) replace		
			
		
*** Binscatter of Share of Months of FSP based on county of birth vs. based 
***	on current counties, by YOB, controlling for age

binscatter share_fsp_0_5 share_cur_fsp_0_5 if 					///
	first_weight_obs == 1 & 									///
	sample_all == 1 [aw = weight],								///
	title(Binscatter: Measures of Childhood FSP Exposure)		///
	subtitle("Full Sample")										///
	ytitle("Share of Childhood (0-5) Exposed to FSP, Birth County")  	///    								///
	xtitle("Share of Childhood (0-5) Exposed to FSP, Current Counties") ///
	ysc(r(0 1)) ylabel(0(0.1)1)  xsc(r(0 1)) xlabel(0(0.1)1)									

graph export ${results}binscatter_full_sample.pdf, as(pdf) replace		


binscatter share_fsp_0_5 share_cur_fsp_0_5 if 					///
	first_weight_obs == 1 & 									///
	sample_all == 1 [aw = weight],								///
	title(Binscatter: Measures of Childhood FSP Exposure)		///
	subtitle("Sample Born During or After 1965")				///
	ytitle("Share of Childhood (0-5) Exposed to FSP, Birth County") 	///    								///
	xtitle("Share of Childhood (0-5) Exposed to FSP, Current Counties") ///
	ysc(r(0 1)) ylabel(0(0.1)1)  xsc(r(0 1)) xlabel(0(0.1)1)									

graph export ${results}binscatter_after_1965.pdf, as(pdf) replace		

binscatter share_fsp_0_5 share_cur_fsp_0_5 if 					///
	first_weight_obs == 1 & 									///
	sample_post_68 == 1 [aw = weight],							///
	title(Binscatter: Measures of Childhood FSP Exposure)		///
	subtitle("Sample Born During or After 1968")				///
	ytitle("Share of Childhood (0-5) Exposed to FSP, Birth County")  	///    								///
	xtitle("Share of Childhood (0-5) Exposed to FSP, Current Counties") ///
	ysc(r(0 1)) ylabel(0(0.1)1)  xsc(r(0 1)) xlabel(0(0.1)1)									
	
graph export ${results}binscatter_after_1968.pdf, as(pdf) replace		
		
		
*** Mover shares by age group and FSP Exposure 
*** Loop over age groups 

gen age_groups = .
replace age_groups = 5 if alt_age == 5
replace age_groups = 10 if alt_age == 10
replace age_groups = 15 if alt_age == 15
label variable age_groups "Age == 5, 10, or 15" 
label define AGE 5 "AGE 5" 10 "AGE 10" 15 "AGE 15", replace 
label values age_group AGE

gen fsp_tag = (share_fsp_0_5 == 1)
label variable fsp_tag "FSP Active in YOB" 

** Share of children with Access to FSP in either their current county or their 
** County of birth

* 1950-1970
estpost tabstat tag_fs_year fsp_tag if sample_all == 1			///							///
		[aw = weight], listwise by(age_groups) columns(statistics)		

esttab . using "${results}means", replace csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in Current or YOB County (1950-1970)") 

* 1965-1970		
estpost tabstat tag_fs_year fsp_tag if sample_post_65 == 1			///							///
		[aw = weight], listwise by(age_groups) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in Current or YOB County (1965-1970)") 

* 1968-1970		
estpost tabstat tag_fs_year fsp_tag if sample_post_68 == 1			///							///
		[aw = weight], listwise by(age_groups) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in Current or YOB County (1968-1970)") 

** Share of children who migrated (Defined different ways)

* 1950-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if sample_all == 1	 											///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. who Moved (1950-1970)") 
		
* 1965-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if sample_post_65 == 1	 										///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. who Moved (1965-1970)") 		
		
* 1968-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if sample_post_68 == 1	 										///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. who Moved (1968-1970)") 	
		
** Share of children (with access to FSP) who migrated (Defined different ways)

* 1950-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 1 & sample_all == 1	 											///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in COB who Moved (1950-1970)") 
		
* 1965-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 1 & sample_post_65 == 1	 										///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in COB who Moved (1965-1970)") 		
		
* 1968-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 1 & sample_post_68 == 1	 										///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in COB who Moved (1968-1970)") 			
		

** Share of children (w/o access to FSP) who migrated (Defined different ways)

* 1950-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 0 & sample_all == 1	 											///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. without FSP in COB who Moved (1950-1970)") 
		
* 1965-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 0 & sample_post_65 == 1	 										///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. without FSP in COB who Moved (1965-1970)") 		
		
* 1968-1970
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 0 & sample_post_68 == 1	 										///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. without FSP in COB who Moved (1968-1970)") 			
		
	
		
		
		
		
		
		
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 1 & alt_yob >= 1968 & alt_yob <= 1970 			///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in COB who Moved (1968-1970)") 
		
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 0	& alt_yob >= 1968 & alt_yob <= 1970 			/// 
		[aw = weight], listwise by(age_groups ) columns(statistics)	
		
esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. w.out FSP in COB who Moved (1968-1970") 		

estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 1 & alt_yob >= 1965 & alt_yob <= 1970 			///
		[aw = weight], listwise by(age_groups ) columns(statistics)		

esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. with FSP in COB who Moved (1965-1970") 
		
estpost tabstat tag_mover tag_mover_to_fsp tag_mover_to_nofsp 			///
		if fsp_tag == 0	& alt_yob >= 1965 & alt_yob <= 1970 			/// 
		[aw = weight], listwise by(age_groups ) columns(statistics)	
		
esttab . using "${results}means", append csv 							///
		main(mean) label unstack not nostar nomtitle nonumber nonotes	///
		title("Share of Obs. w.out FSP in COB who Moved (1965-1970)") 		
